import { Suspense } from "react";
import { useRouter } from "next/router";
import { useQuery, usePaginatedQuery } from "@blitzjs/rpc";
import { Routes, useParam } from "@blitzjs/next";

import Layout from "app/core/layouts/Layout";
import getShortUrl from "app/short-urls/queries/getShortUrl";
import getHitCounters from "app/hit-counters/queries/getHitCounters";
import { CartesianGrid, Line, XAxis, YAxis, LineChart } from "recharts";
import Header from "app/components/Header/Header";
import ShortUrlStatusToggle from "app/short-urls/components/ShortUrlStatusToggle/ShortUrlStatusToggle";
import Main from "app/core/components/Main/Main";
import Link from "next/link";
import ShortUrlLink from "app/short-urls/components/ShortUrlLink/ShortUrlLink";
import Loading from "app/core/components/Loading/Loading";

const ITEMS_PER_PAGE = 100;

const ManageShortUrlView = ({ id }) => {
  const router = useRouter();
  const page = Number(router.query.page) || 0;
  const [shortUrl, { refetch }] = useQuery(getShortUrl, { id });
  const [{ hitCounters, hasMore }] = usePaginatedQuery(getHitCounters, {
    where: { shortUrlId: id },
    orderBy: { date: "desc" },
    skip: ITEMS_PER_PAGE * page,
    take: ITEMS_PER_PAGE,
  });

  const goToPreviousPage = () => router.push({ query: { page: page - 1 } });
  const goToNextPage = () => router.push({ query: { page: page + 1 } });

  return (
    <Main>
      <nav>
        <Link href={Routes.ShortenPage()}>← back to links list</Link>
      </nav>
      <Header>Management</Header>
      <p>
        {shortUrl.url} is shortened to <ShortUrlLink id={id} />
      </p>
      <p>
        Link is {shortUrl.active ? "active" : "deactivated"}.{" "}
        <ShortUrlStatusToggle shortUrl={shortUrl} refetch={refetch} />
      </p>
      <Header>Stats</Header>
      <Chart data={hitCounters} />
      {(hasMore || page !== 0) && (
        <>
          <button disabled={page === 0} onClick={goToPreviousPage}>
            Previous
          </button>
          <button disabled={!hasMore} onClick={goToNextPage}>
            Next
          </button>
        </>
      )}
    </Main>
  );
};

const Chart = ({ data }) => {
  if (data.length === 0) {
    return <kbd>NO HIT COUNT DATA YET</kbd>;
  }
  data.sort((a, b) => (a.date > b.date ? 1 : -1));
  return (
    <LineChart width={500} height={300} data={data}>
      <XAxis dataKey="date" />
      <YAxis />
      <CartesianGrid stroke="#eee" strokeDasharray="5 5" />
      <Line type="monotone" dataKey="count" stroke="#8884d8" />
    </LineChart>
  );
};

const ManageShortUrlPage = () => {
  const id = useParam("id", "string");
  if (!id) {
    return null;
  }
  return (
    <Layout title={id}>
      <main>
        <Suspense fallback={<Loading />}>
          <ManageShortUrlView id={id} />
        </Suspense>
      </main>
    </Layout>
  );
};

ManageShortUrlPage.authenticate = true;

export default ManageShortUrlPage;
